package cz.fidentis.analyst.procrustes;

import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.feature.FeaturePoint;
import cz.fidentis.analyst.icp.IcpTransformation;
import cz.fidentis.analyst.mesh.core.MeshPoint;
import cz.fidentis.analyst.procrustes.exceptions.ProcrustesAnalysisException;
import cz.fidentis.analyst.procrustes.utils.ProcrustesAnalysisUtils;
import org.ejml.simple.SimpleMatrix;
import org.ejml.simple.SimpleSVD;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import javax.vecmath.Point3d;


/**
 * @author Jakub Kolman
 */
public class ProcrustesAnalysis {

    private HumanFace humanFace1;
    private HumanFace humanFace2;

    private List<FeaturePoint> orderedFeaturePointList1;
    private List<FeaturePoint> orderedFeaturePointList2;

    private ProcrustesAnalysisFaceModel faceModel1;
    private ProcrustesAnalysisFaceModel faceModel2;

    private Point3d modelCentroid1;
    private Point3d modelCentroid2;

    // based on some example models this is an approximate centroid of feature points
//    private Point3d centroid  = new Point3d(0, 0, 0);

    /**
     * Constructor
     *
     * @param humanFace1
     * @param humanFace2
     * @throws ProcrustesAnalysisException
     */
    public ProcrustesAnalysis(
            HumanFace humanFace1,
            HumanFace humanFace2) throws ProcrustesAnalysisException {
        this.humanFace1 = humanFace1;
        this.humanFace2 = humanFace2;

        ProcrustesAnalysisFaceModel model1 = new ProcrustesAnalysisFaceModel(humanFace1);
        ProcrustesAnalysisFaceModel model2 = new ProcrustesAnalysisFaceModel(humanFace2);

        if (model1.getFeaturePointsMap().values().size() != model2.getFeaturePointsMap().values().size()) {
            throw new ProcrustesAnalysisException("Lists of feature points do not have the same size");
        }

        this.orderedFeaturePointList1 = new ArrayList<>(model1.getFeaturePointsMap().values());
        this.orderedFeaturePointList2 = new ArrayList<>(model2.getFeaturePointsMap().values());

        if (!featurePointTypesEquivalence(orderedFeaturePointList1, orderedFeaturePointList2)) {
            throw new ProcrustesAnalysisException("Lists of feature points do not have the same feature point types");
        }

        this.faceModel1 = model1;
        this.faceModel2 = model2;

        this.modelCentroid1 = ProcrustesAnalysisUtils.findCentroidOfFeaturePoints(orderedFeaturePointList1);
        this.modelCentroid2 = ProcrustesAnalysisUtils.findCentroidOfFeaturePoints(orderedFeaturePointList2);

    }

    public ProcrustesAnalysisFaceModel getFaceModel1() {
        return faceModel1;
    }

    public ProcrustesAnalysisFaceModel getFaceModel2() {
        return faceModel2;
    }

    public List<IcpTransformation> analyze(ProcrustesAnalysisFaceModel faceModel1, ProcrustesAnalysisFaceModel faceModel2) {
        List<IcpTransformation> transformation = new ArrayList<>();

        superImpose(faceModel1, faceModel2);
        rotate(faceModel1, faceModel2);
        return transformation;
    }

    private Point3d computeCentroidsDistance(Point3d centroid1, Point3d centroid2) {
        double x = (centroid1.x - centroid2.x);
        double y = (centroid1.y - centroid2.y);
        double z = (centroid1.z - centroid2.z);

        Point3d computedCentroid = new Point3d(x, y, z);
        return computedCentroid;
    }

    /**
     * Imposes two face models (lists of feature points and vertices) over each other
     *
     * @param featurePointModel1
     * @param featurePointModel2
     */
    private void superImpose(ProcrustesAnalysisFaceModel featurePointModel1,
                             ProcrustesAnalysisFaceModel featurePointModel2) {

        // moves vertices of face models
        moveModelToSameOrigin(featurePointModel2.getVertices(), featurePointModel2.getFeaturePointsMap());

//        moveVertices(this.modelCentroid1, featurePointModel1.getVertices());
        moveModelToPoint(this.modelCentroid1, featurePointModel1.getVertices(), featurePointModel1.getFeaturePointsMap());
        moveModelToPoint(this.modelCentroid1, featurePointModel2.getVertices(), featurePointModel2.getFeaturePointsMap());
    }

    private void moveModelToSameOrigin(List<MeshPoint> vertices, HashMap<Integer, FeaturePoint> featurePointsMap) {
        Point3d diff = computeCentroidsDistance(modelCentroid1, modelCentroid2);
        moveModelToPoint(diff, vertices, featurePointsMap);
    }

    private void moveModelToPoint(Point3d centroid, List<MeshPoint> vertices, HashMap<Integer, FeaturePoint> featurePointsMap){
        moveVertices(centroid, vertices);
        moveFeaturePointsToVertices(centroid, featurePointsMap);

    }

    private void moveFeaturePointsToVertices(Point3d centroid, HashMap<Integer, FeaturePoint> featurePointsMap) {
        for (FeaturePoint fp: featurePointsMap.values()) {
            fp.getPosition().x = fp.getPosition().x + centroid.x;
            fp.getPosition().y = fp.getPosition().y + centroid.y;
            fp.getPosition().z = fp.getPosition().z + centroid.z;
        }
    }

    /**
     * By rotation of matrices solves orthogonal procrustes problem
     *
     * @param featurePointModel1
     * @param featurePointModel2
     */
    private void rotate(ProcrustesAnalysisFaceModel featurePointModel1,
                        ProcrustesAnalysisFaceModel featurePointModel2) {
        // There is no reason trying to rotate less than 3 elements
        if (this.orderedFeaturePointList1.size() < 3) {
            return;
        }
        SimpleMatrix primaryMatrix = ProcrustesAnalysisUtils.createFeaturePointMatrix(orderedFeaturePointList2);
        SimpleMatrix transposedMatrix = ProcrustesAnalysisUtils.createFeaturePointMatrix(
                orderedFeaturePointList1).transpose();

        SimpleMatrix multipliedMatrix = transposedMatrix.mult(primaryMatrix);
        SimpleSVD<SimpleMatrix> singularValueDecomposition = multipliedMatrix.svd();
        SimpleMatrix transposedU = singularValueDecomposition.getU().transpose();
        SimpleMatrix r = singularValueDecomposition.getV().mult(transposedU);
        primaryMatrix = primaryMatrix.mult(r);

        featurePointModel2.setFeaturePointsMap(
                ProcrustesAnalysisUtils.createFeaturePointMapFromMatrix(primaryMatrix, featurePointModel2));

        rotateVertices(featurePointModel2, primaryMatrix);
    }

    /**
     * Rotates every vertex of model by given matrix
     *
     * @param model
     * @param matrix
     */
    // if rotated vertices are drawn immediately it is better to set them after rotating them all
    // so it would be drawn just once
    private void rotateVertices(ProcrustesAnalysisFaceModel model, SimpleMatrix matrix) {
        if (model.getVertices() != null) {
            int i = 0;
            for (MeshPoint v : model.getVertices()) {
                v.setPosition(ProcrustesAnalysisUtils.rotateVertex(v, matrix));
            }
        }
    }

    /**
     * Calculates new vertices adjusted to the centroid by method {@see findCenteroidOfFeaturePoints},
     * so moved featurepoints would correspond with the same place on face as they did before.
     *  @param centroid
     * @param vertices
     * @return
     */
    private void moveVertices(Point3d centroid, List<MeshPoint> vertices) {
        if (vertices != null && centroid != null) {
//            List<MeshPoint> movedVertices = null;
            for (MeshPoint v : vertices) {
                v.getPosition().x = (v.getPosition().x + centroid.x);
                v.getPosition().y = (v.getPosition().y + centroid.y);
                v.getPosition().z = (v.getPosition().z + centroid.z);
//                float x = (float) (v.getPosition().x + centroid.x);
//                float y = (float) (v.getPosition().y + centroid.y);
//                float z = (float) (v.getPosition().z + centroid.z);
//                Point3d p = new Point3d(x, y, z);
//                v.setPosition(new Point3d(x, y, z));
            }
        } else {
            throw new ProcrustesAnalysisException("Could not compute vertices locations after moving the centroid from model to feature points");
        }
    }

    /**
     * Creates sorted feature point lists and compares them whether they have
     * the same feature point types.
     *
     * @param fpList1
     * @param fpList2
     * @return
     */
    private boolean featurePointTypesEquivalence(List<FeaturePoint> fpList1, List<FeaturePoint> fpList2) {
        return ProcrustesAnalysisUtils.checkFeaturePointsType(fpList1, fpList2);
    }

    /**
     * Calculate scaling ratio of how much the appropriate object corresponding
     * to the second feature point list has to be scale up or shrunk.
     * <p>
     * If returned ratioValue is greater 1 then it means that the second object
     * should be scaled up ratioValue times. If returned ratioValue is smaller 1
     * than the second object should be shrunk.
     *
     * @return ratioValue
     */
    private double calculateScalingValue() {
        double[] distancesOfList1 = ProcrustesAnalysisUtils.calculateMeanDistancesFromOrigin(orderedFeaturePointList1);
        double[] distancesOfList2 = ProcrustesAnalysisUtils.calculateMeanDistancesFromOrigin(orderedFeaturePointList2);

        double[] ratioArray = new double[distancesOfList1.length];
        double ratioValue = 0;

        for (int i = 0; i < distancesOfList1.length; i++) {
            ratioArray[i] += distancesOfList1[i] / distancesOfList2[i];
        }

        for (double ratio : ratioArray) {
            ratioValue += ratio;
        }

        return ratioValue / distancesOfList1.length;
    }

}
