package cz.fidentis.analyst.visitors.mesh.sampling;

import cz.fidentis.analyst.mesh.core.Curvature;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshPoint;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * A relevance-based point sampling using highest curvature.
 * {@see https://graphics.stanford.edu/papers/zipper/zipper.pdf}
 * If curvature is not set in inspected facets, then it is computed and set automatically.
 * This visitor is tread-safe.
 * 
 * @author Radek Oslejsek
 * @author Natalia Bebjakova
 */
public class CurvatureSampling extends PointSampling {
    
    /**
     * Curvature algorithm used for the selection of the top X significant points.
     * 
     * @author Radek Oslejsek
     */
    public enum CurvatureAlg {
        MEAN,
        GAUSSIAN,
        MAX,
        MIN
    }

    private final CurvatureAlg curvatureAlg;
    private List<MeshPoint> allVertices = new ArrayList<>();
    
    /** 
     * Constructor for no point sampling (100 %).
     * 
     * @param cAlg Curvature strategy to be used for the selection of significant points.
     */
    public CurvatureSampling(CurvatureAlg cAlg) {
        this(cAlg, 1.0);
    }

    /**
     * Constructor.
     * 
     * @param cAlg Curvature strategy to be used for the selection of significant points.
     * @param max Maximal number of significant points
     * @throws IllegalArgumentException if the {@code curbatureVisitor} is missing
     */
    public CurvatureSampling(CurvatureAlg cAlg, int max) {
        super(max);
        this.curvatureAlg = cAlg;
    }
    
    /**
     * Constructor.
     * 
     * @param cAlg Curvature strategy to be used for the selection of significant points.
     * @param perc Maximal number of significant points as percents of the original size
     */
    public CurvatureSampling(CurvatureAlg cAlg, double perc) {
        super(perc);
        this.curvatureAlg = cAlg;
    }
    
    @Override
    public void visitMeshFacet(MeshFacet facet) {
        if (facet != null) {
            synchronized(this) {
                if (facet.getVertex(0).getCurvature() == null) {
                    facet.accept(new cz.fidentis.analyst.visitors.mesh.CurvatureCalculator());
                }
                allVertices.addAll(facet.getVertices());
            }
        }
    }
    
    @Override
    public List<MeshPoint> getSamples() {
        return allVertices.stream()
                .sorted((MeshPoint mp1, MeshPoint mp2) -> {
                    Curvature c1 = mp1.getCurvature();
                    Curvature c2 = mp2.getCurvature();
                    switch (curvatureAlg) {
                        case MEAN:
                            return Double.compare(c1.getMean(), c2.getMean());
                        case GAUSSIAN:
                            return Double.compare(c1.getGaussian(), c2.getGaussian());
                        case MAX:
                            return Double.compare(c1.getMaxPrincipal(), c2.getMaxPrincipal());
                        case MIN:
                            return Double.compare(c1.getMinPrincipal(), c2.getMinPrincipal());
                        default:
                            throw new IllegalArgumentException("curvatureAlg");
                    }
                })
                .limit(getNumDownsampledPoints(allVertices.size()))
                .collect(Collectors.toList());
    }
    
    public CurvatureAlg getCurvatureAlg() {
        return this.curvatureAlg;
    }
    
    /**
     * Returns curvatures of selected samples
     * 
     * @return curvatures of selected samples
     */
    //public List<Double> getSampledCurvatures() {
    //    checkAndComputeSignificantPoints();
    //    return significantPoints.stream().map(vc -> vc.curvature).collect(Collectors.toList());
    //}
    
    @Override
    public String toString() {
        return this.curvatureAlg + " curvature " + super.toString();
    }
    
}
