package cz.fidentis.analyst.face;

import cz.fidentis.analyst.Logger;
import cz.fidentis.analyst.feature.FeaturePoint;
import cz.fidentis.analyst.icp.IcpTransformer;
import cz.fidentis.analyst.icp.Quaternion;
import cz.fidentis.analyst.procrustes.ProcrustesAnalysis;
import cz.fidentis.analyst.symmetry.Plane;
import cz.fidentis.analyst.visitors.mesh.sampling.PointSampling;
import java.util.zip.DataFormatException;
import javax.vecmath.Matrix3d;
import javax.vecmath.Matrix4d;
import javax.vecmath.Point3d;
import javax.vecmath.Tuple3d;
import javax.vecmath.Vector3d;

/**
 * A utility class for operations (visitors) applied onto the whole human faces.
 * Currently, transformation operations that affect multiple parts of the human
 * face consistently (mesh, symmetry plane, feature points, etc.) are provided.
 * 
 * @author Radek Oslejsek
 */
public class HumanFaceUtils {
    
    /**
     * Superimpose two faces by applying iterative closest points (ICP) algorithm
     * on their triangular meshes.
     * 
     * @param staticFace A face that remains unchanged. 
     * @param transformedFace A face to be transformed. 
     * @param maxIterations Maximal number of ICP iterations, bigger than zero.
     * Reasonable number seems to be 10.
     * @param scale Whether to scale face as well
     * @param error Acceptable error (a number bigger than or equal to zero).
     * @param samplingStrategy Downsampling strategy
     * @param recomputeKdTree If {@code true} and the k-d tree of the {@code transformedFace} exists, 
     *        the it automatically re-computed. Otherwise, it is simply removed.
     * @return ICP visitor that holds the transformations performed on the {@code transformedFace}.
     */
    public static IcpTransformer alignMeshes(
            HumanFace staticFace, HumanFace transformedFace,
            int maxIterations, boolean scale, double error, 
            PointSampling samplingStrategy,
            boolean recomputeKdTree) {
        
        // transform mesh:
        IcpTransformer icp = new IcpTransformer(
                staticFace.getMeshModel(), 
                maxIterations, 
                scale, 
                error, 
                samplingStrategy
        );
        transformedFace.getMeshModel().compute(icp, true); // superimpose face towards the static face

        // update k-d of transformed face:
        if (transformedFace.hasKdTree()) { 
            if (recomputeKdTree) {
                transformedFace.computeKdTree(true);
            } else {
                transformedFace.removeKdTree();
            }
        }
        
        // update bounding box, which is always present:
        transformedFace.updateBoundingBox();
        
        // transform feature points:
        if (transformedFace.hasFeaturePoints()) {
            icp.getTransformations().values().forEach(trList -> { // List<IcpTransformation> 
                trList.forEach(tr -> { // IcpTransformation
                    for (int i = 0; i < transformedFace.getFeaturePoints().size(); i++) {
                        FeaturePoint fp = transformedFace.getFeaturePoints().get(i);
                        Point3d trPoint = tr.transformPoint(fp.getPosition(), scale);
                        transformedFace.getFeaturePoints().set(i, 
                                new FeaturePoint(trPoint.x, trPoint.y, trPoint.z, fp.getFeaturePointType())
                        );
                    }
                });
            });
        }
        
        // transform symmetry plane:
        if (transformedFace.hasSymmetryPlane()) {
            icp.getTransformations().values().forEach(trList -> { // List<IcpTransformation> 
                trList.forEach(tr -> { // IcpTransformation
                    transformedFace.setSymmetryPlane(transformPlane(
                            transformedFace.getSymmetryPlane(), 
                            tr.getRotation(), 
                            tr.getTranslation(), 
                            tr.getScaleFactor()));
                });
            });
        }
        
        return icp;
    }
    
    /**
     * Transform the face.
     * 
     * @param face Face to be transformed.
     * @param rotation Rotation vector denoting the rotation angle around axes X, Y, and Z.
     * @param translation Translation vector denoting the translation in the X, Y, and Z direction.
     * @param scale Scale factor (1 = no scale).
     * @param recomputeKdTree If {@code true} and the k-d tree of the {@code transformedFace} exists, 
     *        the it automatically re-computed. Otherwise, it is simply removed.
     */
    public static void transformFace(HumanFace face, Vector3d rotation, Vector3d translation, double scale, boolean recomputeKdTree) {
        Quaternion rot = new Quaternion(rotation.x, rotation.y, rotation.z, 1.0);
        
        // update mesh
        face.getMeshModel().getFacets().stream()
                .map(facet -> facet.getVertices())
                .flatMap(meshPoint -> meshPoint.parallelStream())
                .forEach(meshPoint -> {
                    transformPoint(meshPoint.getPosition(), rot, translation, scale);
                    transformNormal(meshPoint.getNormal(), rot);
                });
                    
        
        // update k-d of transformed face:
        if (face.hasKdTree()) { 
            if (recomputeKdTree) {
                face.computeKdTree(true);
            } else {
                face.removeKdTree();
            }
        }
        
        // update bounding box, which is always present:
        face.updateBoundingBox();

        // transform feature points:
        if (face.hasFeaturePoints()) {
            face.getFeaturePoints().parallelStream().forEach(fp -> {
                transformPoint(fp.getPosition(), rot, translation, scale);
            });
        }

        // transform symmetry plane:
        if (face.hasSymmetryPlane()) {
            face.setSymmetryPlane(transformPlane(face.getSymmetryPlane(), rot, translation, scale));
        }
    }
    
    /**
     * Transforms one face so that its symmetry plane fits the symmetry plane of the other face.
     * 
     * @param staticFace a human face that remains unchanged
     * @param transformedFace a human face that will be transformed
     * @param recomputeKdTree  If {@code true} and the k-d tree of the {@code transformedFace} exists, 
     *        the it automatically re-computed. Otherwise, it is simply removed.
     * @param preserveUpDir If {@code false}, then the object can be rotated around the target's normal arbitrarily.
     */
    public static void alignSymmetryPlanes(
            HumanFace staticFace, HumanFace transformedFace, 
            boolean recomputeKdTree, boolean preserveUpDir) {
        
        Plane statPlane = staticFace.getSymmetryPlane();
        Plane tranPlane = transformedFace.getSymmetryPlane();
        
        if (statPlane.getNormal().dot(tranPlane.getNormal()) < 0) {
            tranPlane = tranPlane.flip();
        }
        
        Matrix4d trMat = statPlane.getAlignmentMatrix(tranPlane, preserveUpDir);
        
        // Transform mesh vertices:
        Matrix3d rotMat = new Matrix3d();
        transformedFace.getMeshModel().getFacets().forEach(f -> {
            f.getVertices().stream().forEach(p -> {
                trMat.transform(p.getPosition());
                if (p.getNormal() != null) {
                    trMat.getRotationScale(rotMat);
                    rotMat.transform(p.getNormal());
                    p.getNormal().normalize();
                }
                //transformPoint(p.getPosition(), rotation, translation, 1.0);
                //rotMat.transform(p.getPosition()); // rotate around the plane's normal
            });
        });
        
        // update k-d of transformed face:
        if (transformedFace.hasKdTree()) { 
            if (recomputeKdTree) {
               transformedFace.computeKdTree(true);
            } else {
               transformedFace.removeKdTree();
            }
        }
        
        // update bounding box, which is always present:
        transformedFace.updateBoundingBox();
        
        // Transform feature points:
        if (transformedFace.hasFeaturePoints()) {
            transformedFace.getFeaturePoints().parallelStream().forEach(fp -> {
                trMat.transform(fp.getPosition());
                //transformPoint(fp.getPosition(), rotation, translation, 1.0);
                //rotMat.transform(fp.getPosition()); // rotate around the plane's normal
            });
        }        
        
        // Transform the symmetry plane
        //transformedFace.setSymmetryPlane(new Plane(staticFace.getSymmetryPlane()));
        Point3d p = transformedFace.getSymmetryPlane().getPlanePoint();
        trMat.transform(p);
        transformedFace.setSymmetryPlane(new Plane(p));
    }
    
    /**
     * Superimpose two faces by applying Procrustes on their feature points.
     * Be aware that both faces are transformed into the origin of the coordinate system!
     * 
     * @param firstFace a human face that remains unchanged
     * @param secondFace a human face that will be transformed
     * @param scale Whether to scale faces as well
     * @param recomputeKdTree If {@code true} and the k-d tree of the {@code transformedFace} exists, 
     *        the it automatically re-computed. Otherwise, it is simply removed.
     */
    public static void alignFeaturePoints(
            HumanFace firstFace, HumanFace secondFace, 
            boolean scale, boolean recomputeKdTree) {
        
        ProcrustesAnalysis pa = null;
        try {
            pa = new ProcrustesAnalysis(firstFace, secondFace, scale);
            pa.analyze();
        } catch (DataFormatException e) {
            Logger.print("Procrustes Analysis experienced exception");
            return;
        }
        
        // update k-d of transformed faces:
        if (firstFace.hasKdTree()) { 
            if (recomputeKdTree) {
                firstFace.computeKdTree(true);
            } else {
                firstFace.removeKdTree();
            }
        }
        // update k-d of transformed faces:
        if (secondFace.hasKdTree()) { 
            if (recomputeKdTree) {
                secondFace.computeKdTree(true);
            } else {
                secondFace.removeKdTree();
            }
        }
        
        // update bounding box, which is always present:
        firstFace.updateBoundingBox();
        secondFace.updateBoundingBox();
        
        // transform feature points:
        if (firstFace.hasFeaturePoints()) {
            // TODO
        }
        if (secondFace.hasFeaturePoints()) {
            // TODO
        }
        
        // transform symmetry plane:
        if (firstFace.hasSymmetryPlane()) {
            // TODO
        }
        if (secondFace.hasSymmetryPlane()) {
            // TODO
        }
    }
    
    /**
     * Create rotation matrix from rotation axis and the angle
     * 
     * @param axis rotation axis
     * @param sinAngle cosine of the angle
     * @param cosAngle sin of the angle
     * @return rotation matrix
     */
    protected static Matrix3d rotMatAroundAxis(Vector3d axis, double sinAngle, double cosAngle) {
        Matrix3d matC = new Matrix3d(
                0, -axis.z, axis.y,
                axis.z, 0, -axis.x,
                -axis.y, axis.x, 0
        );
        
        Matrix3d matCC = new Matrix3d(matC);
        matCC.mul(matCC);
        matCC.mul(1.0 - cosAngle);
        
        matC.mul(sinAngle);
        
        Matrix3d rotMat = new Matrix3d();
        rotMat.setIdentity();
        rotMat.add(matC);
        rotMat.add(matCC);
        
        return rotMat;
    }
    
    /**
     * Transform a single 3d point.
     * 
     * @param point point to be transformed
     * @param rotation rotation, can be {@code null}
     * @param translation translation
     * @param scale scale
     */
    protected static void transformPoint(Tuple3d point, Quaternion rotation, Vector3d translation, double scale) {
        Quaternion rotQuat = new Quaternion(point.x, point.y, point.z, 1);

        if (rotation != null) {
            Quaternion rotationCopy = Quaternion.multiply(rotQuat, rotation.getConjugate());
            rotQuat = Quaternion.multiply(rotation, rotationCopy);
        }

        point.set(
                rotQuat.x * scale + translation.x,
                rotQuat.y * scale + translation.y,
                rotQuat.z * scale + translation.z
        );
    }
    
    protected static void transformNormal(Tuple3d normal, Quaternion rotation) {
        if (normal != null) {
            transformPoint(normal, rotation, new Vector3d(0, 0, 0), 1.0); // rotate only
        }
    }

    /**
     * Transforms the whole plane, i.e., its normal and position.
     * 
     * @param plane plane to be transformed
     * @param rot rotation
     * @param translation translation
     * @param scale scale
     * @return transformed plane
     */
    protected static Plane transformPlane(Plane plane, Quaternion rot, Vector3d translation, double scale) {
        Point3d point = new Point3d(plane.getNormal());
        transformNormal(point, rot);
        Plane retPlane = new Plane(point, plane.getDistance());

        // ... then translate and scale a point projected on the rotate plane:
        point.scale(retPlane.getDistance()); // point laying on the rotated plane
        transformPoint(point, null, translation, scale); // translate and scale only
        Vector3d normal = retPlane.getNormal();
        double dist = ((normal.x * point.x) + (normal.y * point.y) + (normal.z * point.z))
                / Math.sqrt(normal.dot(normal)); // distance of tranformed surface point in the plane's mormal direction
        
        return new Plane(retPlane.getNormal(), dist);
    }
    

    
    
}
