package cz.fidentis.analyst;

import cz.fidentis.analyst.face.HumanFace;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;


/**
 * This class encapsulates data for a "comparison project", i.e., a project 
 * enabling analysts to compare and investigate human faces.
 *
 * @author Matej Kovar
 */
public class Project {
    
    private boolean saved = true;
    
    /* Project data (paths to faces, opened tabs..) */
    private ProjectConfiguration cfg = new ProjectConfiguration();
    
    /**
     * Asks whether project is saved or some change was made
     * @return true if project is saved, false otherwise
     */
    public boolean isSaved() {
        return saved;
    }

    public void setSaved(boolean saved) {
        this.saved = saved;
    }

    public ProjectConfiguration getCfg() {
        return cfg;
    }

    public void setCfg(ProjectConfiguration cfg) {
        this.cfg = cfg;
    }
    
    /**
     * Adds new path to project configuration
     * @param path Path to be added
     * @return true if path was successfully added
     */
    public boolean addNewPath(Path path) {
        return this.cfg.addPath(path);
    }
    
    /**
     * Checks whether path is already loaded in project configuration
     * @param path Path to be checked
     * @return true if path was loaded, false otherwise
     */
    public boolean pathLoaded(Path path) {
        return this.cfg.getPaths().contains(path);
    }
    
    /**
     * Adds new FaceToFace tab to project configuration
     * @param name1 String name of first face
     * @param name2 String name of second face
     */
    public void addNewFaceToFaceTabFace(String name1, String name2) {
        this.cfg.addFaceToFaceTabFace(name1, name2);
    }
    
    /**
     * Adds new face to FaceTab
     * @param name String name of face
     */
    public void addNewSingleFaceTabFace(String name) {
        this.cfg.addSingleFaceTabFace(name);
    }
    
    /**
     * Removes FaceTab
     * @param name String name of face
     */
    public void removeFaceTab(String name) {
        this.cfg.removeFaceTab(name);
    }
    
    /**
     * Removes FaceToFace tab
     * @param name1 String name of first face
     * @param name2 String name of second face
     */
    public void removeFaceToFaceTab(String name1, String name2) {
        this.cfg.removeFaceToFaceTab(name1, name2);
    }
       
    /**
     * Loads face from path
     * @param name String name of face
     * @return loaded HumanFace or null
     */
    public HumanFace loadFace(String name) {
        try {
            Logger out = Logger.measureTime();
            Path path = this.getCfg().getPathToFaceByName(name);
            File file = path.toFile();
            HumanFace face = new HumanFace(file, true); // loads also landmarks, if exist
            out.printDuration("Loaded model " + face.getShortName() +" with " + face.getMeshModel().getNumVertices() + " vertices");
            return face;
        } catch (IOException ex) {
            //ex.printStackTrace();
            Logger.print(ex.toString());
        }    
        return null;
    }
   
}
