package cz.fidentis.analyst.symmetry;

import cz.fidentis.analyst.mesh.core.Curvature;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshPoint;
import cz.fidentis.analyst.visitors.mesh.BoundingBox;
import cz.fidentis.analyst.visitors.mesh.CurvatureCalculator;
import cz.fidentis.analyst.visitors.mesh.sampling.PointSampling;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.logging.Level;
import javax.vecmath.Vector3d;

/**
 * A old implementation of the symmetry plane estimator taken from the old FIDENTS.
 * Conceptually, it is similar to the {@link SymmetryEstimatorRobustMesh}.
 * It also uses Gaussian curvature and normal vectors to prune candidate planes 
 * and measure their quality. But some additional criteria are used as well. 
 * Also, the calculation of "votes" differs.
 * This implementation has the following properties:
 * <ul>
 * <li>Fast. A bit faster than the new robust algorithms of {@link SymmetryEstimatorRobust}
 * and much faster that the {@link SymmetryEstimatorRobustMesh}.</li>
 * <li>Best candidate planes are chosen based on the similarity of Gauss curvatures 
 * on both sides of the plane, inverse direction of normal vectors, 
 * and the distance of plane from centroid.</li>
 * <li>Because the space of candidate planes is roughly approximated (several true-false rules), 
 * the final symmetry plane is computed by averaging best candidates (candidates with the most votes).</li>
 * <li>Best results are achieved with Uniform Grid sampling with 200-300 points and averaging turn on.</li>
 * <li>Should return reasonable results also for incomplete faces (was not tested).</li>
 * </ul>
 * <p>
 * This visitor <b>is not thread-safe</b>, i.e., a single instance of the visitor 
 * cannot be used to inspect multiple meshes simultaneously  (sequential inspection is okay). 
 * It because the underlying visitors may not be thread-safe.
 * </p>
 * <p>
 * The main symmetry plane computation is performed by the {@link #getSymmetryPlane()} method,
 * not the {@link #visitMeshFacet(cz.fidentis.analyst.mesh.core.MeshFacet)}.
 * </p>
 * 
 * @author Natalia Bebjakova
 * @author Radek Oslejsek
 */
public class SymmetryEstimatorMesh extends SymmetryEstimator {
    
    public static final double MAX_REL_DISTANCE = 0.01;
    
    /**
     * If {@code true}, then the final symmetry plane is computed by averaging the best candidates.
     * If {@code false}, then the random top candidate is selected.
     */
    public static final boolean AVERAGING = true;
    
    private final PointSampling samplingStrategy;
    private final int samplingLimit;
    private final BoundingBox bbVisitor = new BoundingBox();
    
    private Plane symmetryPlane;
    
    /**
     * Constructor.
     * 
     * @param samplingStrategy Downsampling strategy. Must not be {@code null}
     * @param samplingLimit Desired number of samples for udenrsampling
     * @throws IllegalArgumentException if some input parameter is missing
     */
    public SymmetryEstimatorMesh(PointSampling samplingStrategy, int samplingLimit) {
        if (samplingStrategy == null) {
            throw new IllegalArgumentException("samplingStrategy");
        }
        this.samplingStrategy = samplingStrategy;
        this.samplingLimit = samplingLimit;
    }
    
    @Override
    public boolean isThreadSafe() {
        return false;
    }
    
    @Override
    public void visitMeshFacet(MeshFacet facet) {
        // We need vertex normals for the plane computation
        synchronized (this) {
            if (!facet.hasVertexNormals()) {
                facet.calculateVertexNormals();
            }
        }
        
        // We need Gaussian curvatures
        synchronized (this) {
            if (facet.getVertex(0).getCurvature() == null) {
                facet.accept(new CurvatureCalculator());
            }
        }
        
        // We want to downsample the mesh
        samplingStrategy.visitMeshFacet(facet);
        
        // We need bounding box to compute max distance
        bbVisitor.visitMeshFacet(facet);
    }
    
    /**
     * Computes the symmetry plane concurrently.The plane is computed only once, then the same instance is returned.
     * 
     * @return Symmetry plane
     */
    public Plane getSymmetryPlane() {
        return getSymmetryPlane(true);
    }
    
    /**
     * Computes and returns the symmetry plane. The plane is computed only once, 
     * then the same instance is returned.
     * 
     * @param concurrently If {@code true}, then parallel computation is used utilizing all CPU cores.
     * @return the symmetry plane or {@code null}
     */
    public Plane getSymmetryPlane(boolean concurrently) {
        if (symmetryPlane == null) {
            this.calculateSymmetryPlane(concurrently);
        }
        return symmetryPlane;
    }
    
    /**
     * Calculates the symmetry plane.
     * @param concurrently If {@code true}, then parallel computation is used utilizing all CPU cores.
     * Otherwise, the computation is sequential.
     */
    protected void calculateSymmetryPlane(boolean concurrently) {
        final List<Plane> planes = new ArrayList<>();
        final double maxDistance = bbVisitor.getBoundingBox().getDiagonalLength() * MAX_REL_DISTANCE;
        
        samplingStrategy.setRequiredSamples(samplingLimit);
        List<MeshPoint> sigVertices = samplingStrategy.getSamples();
        SymmetryCache cache = new SymmetryCache(sigVertices);
        
        /*
         * Sequential computation
         */
        if (!concurrently) {
            int maxVotes = 0;
            for (int i = 0; i < sigVertices.size(); i++) {
                for (int j = 0; j < sigVertices.size(); j++) {
                    CandidatePlane newPlane;
                    try {
                        newPlane = new CandidatePlane(sigVertices, cache, i, j, maxDistance);
                    } catch(UnsupportedOperationException ex) {
                        continue;
                    }
                    maxVotes = checkAndUpdatePlanes(planes, newPlane, maxVotes);
                }
            }
            setSymmetryPlane(planes);
            return;
        }
        
        /*
         * Concurrent computation
         */
        
        // Initiate structure for concurrent computation:
        ExecutorService executor = Executors.newFixedThreadPool(Runtime.getRuntime().availableProcessors());
        final List<Future<CandidatePlane>> results = new ArrayList<>(sigVertices.size() * sigVertices.size());
        
        // Compute candidate planes concurrently:
        for (int i = 0; i < sigVertices.size(); i++) {
            for (int j = 0; j < sigVertices.size(); j++) {
                int finalI = i;
                int finalJ = j;
                
                Callable<CandidatePlane> worker = new  Callable<CandidatePlane>() {
                    @Override
                    public CandidatePlane call() throws Exception {
                        try {
                            return new CandidatePlane(sigVertices, cache, finalI, finalJ, maxDistance);
                        } catch(UnsupportedOperationException ex) {
                            return null;
                        }
                    }
                };
                
                results.add(executor.submit(worker));
            }
        }
        
        // Wait until all symmetry planes are computed:
        executor.shutdown();
        while (!executor.isTerminated()){}
        
        // Get the best-fitting planes:
        try {
            int maxVotes = 0;
            for (Future<CandidatePlane> res: results) {
                CandidatePlane newPlane = res.get();
                if (newPlane != null) {
                    maxVotes = checkAndUpdatePlanes(planes, newPlane, maxVotes);
                }
            }
        } catch (final InterruptedException | ExecutionException ex) {
            java.util.logging.Logger.getLogger(SymmetryEstimatorMesh.class.getName()).log(Level.SEVERE, null, ex);
        }
        
        setSymmetryPlane(planes);
    }
    
    protected int checkAndUpdatePlanes(List<Plane> planes, CandidatePlane newPlane, int maxVotes) {
        if (newPlane.votes > maxVotes) {
            planes.clear();
            planes.add(newPlane);
            return newPlane.votes;
        } else if (newPlane.votes == maxVotes) {
            planes.add(newPlane);
        }
        return maxVotes;
    }

    protected void setSymmetryPlane(List<Plane> planes) {
        if (AVERAGING && !planes.isEmpty()) {
            symmetryPlane = new Plane(planes);
        } else {
            symmetryPlane = planes.isEmpty() ? null : planes.get(0);
        }
    }
    
    /**
     * Symmetry plane with votes used for the decision about symmetry estimate
     * of 3D models.
     *
     * @author Natalia Bebjakova
     *
     */
    protected class CandidatePlane extends Plane implements Comparable<CandidatePlane> {

        public static final double MIN_ANGLE_COS = 0.985;
        public static final double AVG_CURVATURE_MULTIPLICATOR = 0.01;

        private int votes;

        /**
         * Constructor.
         *
         * @param vertices Mesh vertices
         * @param cache Precomputed values form mesh vertices
         * @param i index of the first significant point used for the plane
         * construction
         * @param j index of the second significant point used for the plane
         * construction
         * @param maxDist Distance limit
         * @throws UnsupportedOperationException if the symmetry plane cannot be
         * created
         */
        protected CandidatePlane(List<MeshPoint> vertices, SymmetryCache cache, int i, int j, double maxDist) throws UnsupportedOperationException {
            if (i == j) {
                throw new UnsupportedOperationException();
            }
            setNormAndDist(vertices, cache, i, j);
            computeVotes(vertices, cache, maxDist);
        }

        /**
         *
         * @param vertices Vertices
         * @param cache Cache
         * @param i Index of the first point
         * @param j Index of the second point
         */
        private void setNormAndDist(List<MeshPoint> vertices, SymmetryCache cache, int i, int j) {
            MeshPoint meshPointI = vertices.get(i);
            MeshPoint meshPointJ = vertices.get(j);

            // accept only points with similar Gaussian curvature
            if (!similarCurvature(meshPointI.getCurvature(), meshPointJ.getCurvature(), cache.avgGaussianCurvature * AVG_CURVATURE_MULTIPLICATOR)) {
                throw new UnsupportedOperationException();
            }

            Vector3d planeNormal = new Vector3d(meshPointI.getPosition());
            planeNormal.sub(meshPointJ.getPosition());
            planeNormal.normalize();

            // accpect only point pair with oposite normals along with the plane normal:
            double normCos = cache.normCosVec[i][j].dot(planeNormal);
            if (Math.abs(normCos) < MIN_ANGLE_COS) {
                throw new UnsupportedOperationException();
            }

            setDistance(-planeNormal.dot(cache.avgPos[i][j]));
            setNormal(planeNormal);
        }

        /**
         * Computes votes for given plane
         *
         * @param sigPoints Mesh vertices with the most significant curvature
         * @param maxDist Distance limit
         */
        private void computeVotes(List<MeshPoint> vertices, SymmetryCache cache, double maxDist) {
            normalize();
            Vector3d normal = getNormal();
            double d = getDistance();

            for (int i = 0; i < vertices.size(); i++) {
                for (int j = 0; j < vertices.size(); j++) {
                    if (i == j) {
                        continue;
                    }

                    if (!similarCurvature(vertices.get(i).getCurvature(), vertices.get(j).getCurvature(), cache.avgGaussianCurvature * AVG_CURVATURE_MULTIPLICATOR)) {
                        continue;
                    }

                    double normCos = cache.normCosVec[i][j].dot(normal);
                    if (Math.abs(normCos) < MIN_ANGLE_COS) {
                        continue;
                    }

                    Vector3d vec = new Vector3d(vertices.get(i).getPosition());
                    vec.sub(vertices.get(j).getPosition());
                    vec.normalize();
                    double cos = vec.dot(normal);
                    if (Math.abs(cos) < MIN_ANGLE_COS) {
                        continue;
                    }

                    if (Math.abs(normal.dot(cache.avgPos[i][j]) + d) <= maxDist) {
                        votes++;
                    }
                }
            }
        }

        private boolean similarCurvature(Curvature gi, Curvature gj, double gh) {
            if (gi == null || gj == null || gi.getGaussian() == Double.NaN || gj.getGaussian() == Double.NaN) {
                return true; // can't decide => continue in the computation
            }
            return (gi.getGaussian() * gj.getGaussian() > 0)
                    && (Math.abs(gi.getGaussian()) >= gh)
                    && (Math.abs(gj.getGaussian()) >= gh);
        }

        /**
         * Enables to compare two approximate planes due to number of votes
         *
         * @param other plane to be compared
         * @return number that decides which plane has more votes
         */
        @Override
        public int compareTo(CandidatePlane other) {
            return Integer.compare(votes, other.votes);
        }

        @Override
        public String toString() {
            return this.getNormal() + " " + getDistance() + " " + votes;
        }
    }
    
    
    /*******************************************************************
     * Precomputed values for the symmetry plane estimation.
     *
     * @author Radek Oslejsek
     */
    protected class SymmetryCache {

        private Vector3d[][] normCosVec;
        private Vector3d[][] avgPos;
        private double       avgGaussianCurvature;

        /**
         * Constructor.
         *
         * @param vertices Mesh vertices
         */
        protected SymmetryCache(List<MeshPoint> vertices) {
            if (vertices == null || vertices.isEmpty()) {
                throw new IllegalArgumentException("points");
            }

            int size = vertices.size();

            normCosVec = new Vector3d[size][size];
            avgPos = new Vector3d[size][size];

            int counter = 0;
            for (int i = 0; i < size; i++) {
                MeshPoint meshPointI = vertices.get(i);

                Curvature gi = meshPointI.getCurvature();
                if (gi != null && gi.getGaussian() != Double.NaN) {
                    avgGaussianCurvature += gi.getGaussian();
                    counter++;
                }

                for (int j = 0; j < vertices.size(); j++) {
                    MeshPoint meshPointJ = vertices.get(j);

                    Vector3d ni = new Vector3d(meshPointI.getNormal());
                    ni.sub(meshPointJ.getNormal());
                    ni.normalize();
                    normCosVec[i][j] = ni;

                    Vector3d avrg = new Vector3d(meshPointI.getPosition());
                    Vector3d aux = new Vector3d(meshPointJ.getPosition());
                    avrg.add(aux);
                    avrg.scale(0.5);
                    avgPos[i][j] = avrg;
                }
            }
            if (counter != 0) {
                avgGaussianCurvature /= counter;
            } else {
                avgGaussianCurvature = Double.NaN;
            }
        }
    }
}
